<?php
session_start();
date_default_timezone_set('Europe/Amsterdam');

$logDir = __DIR__ . '/logs';
$bestanden = glob("$logDir/*.json");
$project_opties = [];
$alle_logs = [];
$totaal_minuten = 0;

setlocale(LC_TIME, 'nl_NL.UTF-8');

// 🔎 Filters ophalen
$filter_project = $_GET['project'] ?? '';
$periode = $_GET['periode'] ?? '';
$filter_vanaf = $_GET['vanaf'] ?? '';
$filter_tot = $_GET['tot'] ?? '';

// 🔄 Periode vertaalt zich naar vanaf/tot datums
switch ($periode) {
    case 'vandaag':
        $filter_vanaf = $filter_tot = date('Y-m-d');
        break;
    case 'deze_week':
        $filter_vanaf = date('Y-m-d', strtotime('monday this week'));
        $filter_tot = date('Y-m-d');
        break;
    case 'deze_maand':
        $filter_vanaf = date('Y-m-01');
        $filter_tot = date('Y-m-t');
        break;
    case 'vorige_maand':
        $filter_vanaf = date('Y-m-01', strtotime('first day of last month'));
        $filter_tot = date('Y-m-t', strtotime('last day of last month'));
        break;
}

foreach ($bestanden as $bestand) {
    $project = basename($bestand, '.json');
    $project_opties[] = $project;

    $data = json_decode(file_get_contents($bestand), true);
    if (!is_array($data)) continue;

    foreach ($data as $index => $entry) {
        $startDT = DateTime::createFromFormat('Y-m-d H:i:s', $entry['start']);
        $endDT = DateTime::createFromFormat('Y-m-d H:i:s', $entry['end']);

        if ($filter_project && $filter_project !== $project) continue;
        if ($filter_vanaf && $startDT < new DateTime($filter_vanaf)) continue;
        if ($filter_tot && $startDT > new DateTime($filter_tot . ' 23:59:59')) continue;

        $duur_min = (int)$entry['duration_minutes'];
        $totaal_minuten += $duur_min;

        $alle_logs[] = [
            'project' => $project,
            'index' => $index,
            'start' => $startDT->format('d-m-Y H:i'),
            'end' => $endDT->format('d-m-Y H:i'),
            'duur' => floor($duur_min / 60) . 'u ' . ($duur_min % 60) . 'm',
            'beschrijving' => $entry['description']
        ];
    }
}

usort($alle_logs, fn($a, $b) => strtotime($a['start']) <=> strtotime($b['start']));
$totaal_tekst = floor($totaal_minuten / 60) . 'u ' . ($totaal_minuten % 60) . 'm';
?>

<!DOCTYPE html>
<html lang="nl">
<head>
    <meta charset="UTF-8">
    <title>Stats logs</title>
    <link rel="stylesheet" href="assets/style.css">
</head>
<body>

<?php include 'includes/menu.php'; ?>

<h2>📋 Stats logs & changes</h2>

<form method="get" style="margin-bottom: 20px;">
    <label for="project">Project:</label>
    <select name="project" id="project">
        <option value="">-- All projects --</option>
        <?php foreach ($project_opties as $optie): ?>
            <option value="<?= htmlspecialchars($optie) ?>" <?= $filter_project === $optie ? 'selected' : '' ?>>
                <?= htmlspecialchars($optie) ?>
            </option>
        <?php endforeach; ?>
    </select>

    <label for="periode">Periode:</label>
    <select name="periode" id="periode" onchange="this.form.submit()">
        <option value="">-- Geen snelle filter --</option>
        <option value="vandaag" <?= $periode === 'vandaag' ? 'selected' : '' ?>>Today</option>
        <option value="deze_week" <?= $periode === 'deze_week' ? 'selected' : '' ?>>This week</option>
        <option value="deze_maand" <?= $periode === 'deze_maand' ? 'selected' : '' ?>>This month</option>
        <option value="vorige_maand" <?= $periode === 'vorige_maand' ? 'selected' : '' ?>>Last month</option>
    </select>

    <label for="vanaf">From:</label>
    <input type="date" name="vanaf" id="vanaf" value="<?= htmlspecialchars($filter_vanaf) ?>">

    <label for="tot">T/m:</label>
    <input type="date" name="tot" id="tot" value="<?= htmlspecialchars($filter_tot) ?>">

    <button type="submit">Filter toepassen</button>
</form>

<?php if (count($alle_logs) === 0): ?>
    <p>⚠️ Geen resultaten gevonden met deze filters.</p>
<?php else: ?>
    <table>
        <thead>
            <tr>
                <th>Project</th>
                <th>Start</th>
                <th>End</th>
                <th>Time</th>
                <th>Note</th>
                <th>Actions</th>
            </tr>
        </thead>
        <tbody>
            <?php foreach ($alle_logs as $log): ?>
                <tr>
                    <td><?= htmlspecialchars($log['project']) ?></td>
                    <td><?= $log['start'] ?></td>
                    <td><?= $log['end'] ?></td>
                    <td><?= $log['duur'] ?></td>
                    <td><?= nl2br(htmlspecialchars($log['beschrijving'])) ?></td>
                    <td><a href="bewerken.php?project=<?= urlencode($log['project']) ?>&index=<?= $log['index'] ?>">✏️ Edit</a></td>
                </tr>
            <?php endforeach; ?>
        </tbody>
    </table>

    <p><strong>🕒 Total time:</strong> <?= $totaal_tekst ?></p>
<?php endif; ?>

</body>
</html>
